<?php

namespace WPAdminify\Pro\RedirectUrls;

use WPAdminify\Pro\RedirectUrls\RedirectUrlsModel;
use WPAdminify\Pro\RedirectUrls\RedirectUrlsSettings;
use WPAdminify\Inc\Utils;

// no direct access allowed
if (!defined('ABSPATH'))  exit;

/**
 * WPAdminify
 * @package Redirect URLs
 *
 * @author Jewel Theme <support@jeweltheme.com>
 */

class RedirectUrls extends RedirectUrlsModel
{
	public $url;
	public $login_redirect_slug;
	public $redirect_admin_url;
	public $old_login_page = false;

	public function __construct()
	{
		$this->url                 = WP_ADMINIFY_URL . 'Inc/Modules/RedirectUrls';
		$this->options             = (new RedirectUrlsSettings())->get('redirect_urls_options');

		$this->login_redirect_slug = !empty($this->options['new_login_url']) ? $this->options['new_login_url'] : '';
		$this->redirect_admin_url = !empty($this->options['redirect_admin_url']) ? $this->options['redirect_admin_url'] : '';

		// Defend wp-admin if the slug is set in the setting.
		if ($this->redirect_admin_url) {
			add_action('wp_loaded', array($this, 'defend_wp_admin'));
		}

		// Stop if custom login slug is not set.
		if (!$this->login_redirect_slug) {
			return;
		}
		add_action('init', [$this, 'change_url'], 99999);
		add_action('wp_loaded', [$this, 'defend_wp_login']);
		add_action('site_url', [$this, 'site_url'], 10, 4);
		add_action('network_site_url', [$this, 'network_site_url'], 10, 3);
		add_action('wp_redirect', [$this, 'wp_redirect'], 10, 2);
		add_action('site_option_welcome_email', [$this, 'welcome_email']);

		remove_action('template_redirect', 'wp_redirect_admin_locations', 99999);
	}

	/**
	 * Change url.
	 */
	public function change_url()
	{
		if (!$this->login_redirect_slug) {
			return;
		}

		global $pagenow;

		$uri = rawurldecode($_SERVER['REQUEST_URI']);

		$has_signup_slug   = false !== stripos($uri, 'wp-signup') ? true : false;
		$has_activate_slug = false !== stripos($uri, 'wp-activate') ? true : false;

		if (!is_multisite() && ($has_signup_slug || $has_activate_slug)) {
			return;
		}

		$request      = wp_parse_url($uri);
		$request_path = isset($request['path']) ? untrailingslashit($request['path']) : '';


		$using_permalink = get_option('permalink_structure') ? true : false;

		$has_new_slug    = isset($_GET[$this->login_redirect_slug]) && $_GET[$this->login_redirect_slug] ? true : false;
		$has_old_slug    = false !== stripos($uri, 'wp-login.php') ? true : false;

		$has_register_slug = false !== stripos($uri, 'wp-register.php') ? true : false;

		if (!is_admin() && ($has_old_slug || site_url('wp-login', 'relative') === $request_path)) {
			$pagenow = 'index.php';
			$this->old_login_page = true;
			$_SERVER['REQUEST_URI']  = $this->is_trailingslashit('/' . str_repeat('-/', 10));
		} elseif (site_url($this->login_redirect_slug, 'relative') === $request_path || (!$using_permalink && $has_new_slug)) {
			// If current page is new login page, let WordPress know if this is a login page.
			$pagenow = 'wp-login.php';
		} elseif (!is_admin() && ($has_register_slug || site_url('wp-register', 'relative') === $request_path)) {
			$pagenow = 'index.php';

			$this->old_login_page = true;
			$_SERVER['REQUEST_URI']  = $this->is_trailingslashit('/' . str_repeat('-/', 10));
		}
	}

	/**
	 * Defend wp-admin
	 */
	public function defend_wp_admin()
	{

		if (isset($_GET['action']) && 'postpass' === $_GET['action'] && isset($_GET['post_password'])) {
			return;
		}

		global $pagenow;

		$request      = wp_parse_url(rawurldecode($_SERVER['REQUEST_URI']));
		$request_path = isset($request['path']) ? $request['path'] : '';

		if (is_admin() && !is_user_logged_in() && !wp_doing_ajax() && 'admin-post.php' !== $pagenow && '/wp-admin/options.php' !== $request_path) {
			wp_safe_redirect($this->admin_redirect_url());
			exit;
		}
	}

	/**
	 * wp-admin redirect url
	 *
	 * @return string
	 */
	public function admin_redirect_url()
	{

		$redirect_url = site_url($this->redirect_admin_url);

		if (get_option('permalink_structure')) {
			return $this->is_trailingslashit($redirect_url);
		}

		return $redirect_url;
	}

	/**
	 * Return a string with or without trailing slash based on permalink structure.
	 */
	public function is_trailingslashit($string)
	{
		$use_trailingslash = '/' === substr(get_option('permalink_structure'), -1, 1) ? true : false;
		return ($use_trailingslash ? trailingslashit($string) : untrailingslashit($string));
	}

	/**
	 * Defend wp-login.php based on the setting.
	 */
	public function defend_wp_login()
	{
		if (isset($_GET['action']) && 'postpass' === $_GET['action'] && isset($_GET['post_password'])) {
			return;
		}

		global $pagenow;

		$request      = wp_parse_url(rawurldecode($_SERVER['REQUEST_URI']));
		$request_path = $request['path'];

		$query_string     = isset($_SERVER['QUERY_STRING']) ? $_SERVER['QUERY_STRING'] : '';
		$add_query_string = $query_string ? '?' . $query_string : '';

		if ('wp-login.php' === $pagenow && $request_path !== $this->is_trailingslashit($request_path) && get_option('permalink_structure')) {
			wp_safe_redirect($this->is_trailingslashit($this->new_login_url()) . $add_query_string);
			exit;
		} elseif ($this->old_login_page) {
			$referer  = wp_get_referer();
			$referers = wp_parse_url($referer);

			$referer_contains_activate_url = false !== stripos($referer, 'wp-activate.php') ? true : false;

			if ($referer_contains_activate_url && !empty($referers['query'])) {
				parse_str($referers['query'], $referer_queries);

				$signup_key           = $referer_queries['key'];
				$wpmu_activate_signup = wpmu_activate_signup($signup_key);

				@require_once WPINC . '/ms-functions.php';

				if (!empty($signup_key) && is_wp_error($wpmu_activate_signup)) {
					if ('already_active' === $wpmu_activate_signup->get_error_code() || 'blog_taken' === $wpmu_activate_signup->get_error_code()) {
						wp_safe_redirect($this->new_login_url() . $add_query_string);
						exit;
					}
				}
			}

			$this->wp_template_loader();
		} elseif ('wp-login.php' === $pagenow) {
			$redirect_to           = admin_url();
			$requested_redirect_to = '';

			if (isset($_REQUEST['redirect_to'])) {
				$requested_redirect_to = $_REQUEST['redirect_to'];
			}

			if (is_user_logged_in()) {
				$user = wp_get_current_user();

				if (!isset($_REQUEST['action'])) {
					wp_safe_redirect($redirect_to);
					exit;
				}
			}

			// Prevent warnings in wp-login.php file by providing these globals.
			global $user_login, $error, $iterim_login, $action;

			@require_once ABSPATH . 'wp-login.php';
			exit;
		}
	}


	/**
	 * Filter site_url.
	 *
	 * @see https://developer.wordpress.org/reference/hooks/site_url/
	 *
	 */
	public function site_url($url, $path, $scheme, $blog_id)
	{

		return $this->filter_old_login_page($url, $scheme);
	}


	/**
	 * Filter old login page.
	 *
	 * @param string $scheme Scheme to give the site URL context. Accepts 'http', 'https', 'login', 'login_post', 'admin', 'relative' or null.
	 *
	 */
	public function filter_old_login_page($url, $scheme = null)
	{

		if (false !== stripos($url, 'wp-login.php?action=postpass')) {
			return $url;
		}

		$url_contains_old_login_url     = false !== stripos($url, 'wp-login.php') ? true : false;
		$referer_contains_old_login_url = false !== stripos(wp_get_referer(), 'wp-login.php') ? true : false;

		if ($url_contains_old_login_url && !$referer_contains_old_login_url) {
			if (is_ssl()) {
				$scheme = 'https';
			}

			$url_parts = explode('?', $url);

			if (isset($url_parts[1])) {
				parse_str($url_parts[1], $args);

				if (isset($args['login'])) {
					$args['login'] = rawurlencode($args['login']);
				}

				$url = add_query_arg($args, $this->new_login_url($scheme));
			} else {
				$url = $this->new_login_url($scheme);
			}
		}

		return $url;
	}

	/**
	 * Filter network_site_url.
	 *
	 * @see https://developer.wordpress.org/reference/hooks/network_site_url/
	 *
	 */
	public function network_site_url($url, $path, $scheme)
	{

		return $this->filter_old_login_page($url, $scheme);
	}

	/**
	 * Filter wp_redirect.
	 *
	 * @see https://developer.wordpress.org/reference/hooks/wp_redirect/
	 *
	 */
	public function wp_redirect($location, $status)
	{

		return $this->filter_old_login_page($location);
	}

	/**
	 * WordPress template loader.
	 *
	 * @return void
	 */
	public function wp_template_loader()
	{

		global $pagenow;

		$pagenow = 'index.php';

		if (!defined('WP_USE_THEMES')) {
			define('WP_USE_THEMES', true);
		}

		wp();

		require_once ABSPATH . WPINC . '/template-loader.php';

		exit;
	}

	/**
	 * Get new login url.
	 *
	 * @param string|null $scheme Scheme to give the site URL context. Accepts 'http', 'https', 'login', 'login_post', 'admin', 'relative' or null.
	 */
	public function new_login_url($scheme = null)
	{

		$login_url = site_url($this->login_redirect_slug, $scheme);

		if (get_option('permalink_structure')) {
			return $this->is_trailingslashit($login_url);
		} else {
			return home_url('/', $scheme) . '?' . $this->login_redirect_slug;
		}

		return $login_url;
	}
}
